/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */

package org.apache.directory.studio.valueeditors.oid;


import org.apache.directory.api.ldap.model.schema.syntaxCheckers.OidSyntaxChecker;
import org.apache.directory.studio.connection.core.Utils;
import org.apache.directory.studio.ldapbrowser.core.model.IValue;
import org.apache.directory.studio.valueeditors.AbstractInPlaceStringValueEditor;


/**
 * Implementation of IValueEditor for syntax 1.3.6.1.4.1.1466.115.121.1.38 
 * (OID syntax). 
 * 
 * Currently only the getDisplayXXX() methods are implemented.
 * For modification the raw string must be edited.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class InPlaceOidValueEditor extends AbstractInPlaceStringValueEditor
{
    /**
     * {@inheritDoc}
     */
    public String getDisplayValue( IValue value )
    {
        String displayValue = super.getDisplayValue( value );

        if ( !showRawValues() )
        {
            String description = Utils.getOidDescription( displayValue );

            if ( description != null )
            {
                displayValue = displayValue + " (" + description + ")"; //$NON-NLS-1$ //$NON-NLS-2$
            }
        }

        return displayValue;
    }


    @Override
    public Object getRawValue( IValue value )
    {
        Object rawValue = super.getRawValue( value );

        // DIRSTUDIO-1216: allows relaxed OID syntax with underscore, e.g. for Oracle or DirX
        if ( rawValue instanceof String
            && OidSyntaxChecker.INSTANCE.isValidSyntax( ( ( String ) rawValue ).replace( "_", "-" ) ) )
        {
            return rawValue;
        }
        else
        {
            return null;
        }
    }
}
