/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra;

/**
 * A place to define all the configuration constants that can be used to
 * configure parts of Orchestra core.
 * 
 * @since 1.1
 */
public final class CoreConfig
{
    private static final String BASE = CoreConfig.class.getName();

    /**
     * Value "org.apache.myfaces.orchestra.CoreConfig:SERIALIZE_REQUESTS".
     * <p>
     * Controls whether two threads are prevented from accessing the same
     * ConversationContext concurrently, or not. This is generally a bad idea as it
     * can lead to ugly race conditions, so it is TRUE by default.
     * <p>
     * It is possible for multiple requests associated with the same http session to be
     * received concurrently. By default, a servlet engine simply processes all requests
     * concurrently in different threads. However that can cause all sorts of unexpected
     * problems with objects in scopes that are visible to both threads (session-scoped
     * and conversation-scoped objects).
     * <p>
     * When enabled, this option will block the thread for any request which will
     * access the same Orchestra ConversationContext as a thread that is already
     * running, and release it only when the earlier request has completed. This protects
     * all orchestra conversation-scoped objects from concurrent access, but
     * <i>does not</i> protect any session-scoped objects.
     * <p>
     * When using Orchestra, it is recommended that session-scoped objects are avoided
     * conversation-scoped beans used instead. If there is no session-scoped data in
     * use by an application then it is safe to allow concurrent requests to the
     * same http session. If you do wish to protect normal session-scoped objects as
     * well, then the standard solution is to write a filter that uses standard java
     * synchronisation on a session-scoped object, taking the lock on request entry
     * and releasing it on request exit.
     * <p>
     * Note that the expression "serialize requests" as used here means requests are
     * processed serially, ie one after the other rather than concurrently. This has
     * nothing to do with java.io.Serializable.
     */
    public static final String SERIALIZE_REQUESTS = BASE + ":SERIALIZE_REQUESTS";

    /**
     * Value "org.apache.myfaces.orchestra.CoreConfig:CLEANUP_CONNECTIONS".
     * <p>
     * Controls whether Orchestra should monitor the set of JDBC connections borrowed
     * during a request, and clean up any that are still open at the end of the request.
     * <p>
     * Orchestra provides a special DataSource wrapper that can be configured for any
     * datasource used by the application. The datasource simply wraps all real Connection
     * objects it returns in a proxy, and keeps track of them. At the end of each request,
     * any connections borrowed by this thread but not returned can be returned
     * automatically. See ConnectionManagerDataSource for more details.
     * <p>
     * If this special DataSource is not configured, then checking for unreleased
     * connections is harmless, and always finds nothing. Therefore this option is
     * enabled by default. 
     */
    public static final String CLEANUP_CONNECTIONS = BASE + ":CLEANUP_CONNECTIONS";

    /**
     * Value "org.apache.myfaces.orchestra.CoreConfig:CONVERSATION_MESSAGER".
     * <p>
     * Controls what class is used to present Orchestra-related error messages to
     * the user. When not specified, the default implementation for the current 
     * FrameworkAdapter (eg JsfFrameworkAdapter) is used.
     * <p>
     * See class ConversationMessager for further details.
     */
    public static final String CONVERSATION_MESSAGER = BASE + ":CONVERSATION_MESSAGER";
}